<?php

declare(strict_types=1);

namespace Skyboard\Infrastructure\Packs;

final class PackLoader
{
    public function __construct(
        private readonly SystemTagRegistry $tags,
        private readonly UiSlotRegistry $ui,
        private readonly DatasetStore $datasets,
        private readonly RulesCatalog $rules,
        private readonly MccValidator $validator = new MccValidator()
    ) {
    }

    public function load(array $enabledPackIds, string $packsDir): void
    {
        foreach ($enabledPackIds as $packId) {
            $manifestPath = $packsDir . '/' . $packId . '/manifest.mcc.json';
            
            if (!file_exists($manifestPath)) {
                error_log("Pack manifest not found: $manifestPath");
                continue;
            }

            $content = file_get_contents($manifestPath);
            if ($content === false) {
                error_log("Cannot read pack manifest: $manifestPath");
                continue;
            }

            $mcc = json_decode($content, true);
            if (!is_array($mcc)) {
                error_log("Invalid JSON in pack manifest: $manifestPath");
                continue;
            }

            // Validation du manifest
            if (!$this->validator->isValid($mcc)) {
                $errors = $this->validator->validate($mcc);
                error_log("Invalid MCC manifest for pack $packId: " . implode(', ', $errors));
                continue;
            }

            // Enregistrement des contributions
            $this->tags->register($packId, $mcc['tags'] ?? []);
            $this->ui->register($packId, $mcc['ui'] ?? []);
            $this->datasets->register($packId, $packsDir, $mcc['datasets'] ?? []);
            $this->rules->registerPack($packId, $mcc['rules'] ?? []);
        }
    }
}
