<?php

declare(strict_types=1);

namespace Skyboard\Infrastructure\Packs;

use Skyboard\Domain\Rules\RuleDefinition;

final class RulesCatalog
{
    /** @var array<string, list<RuleDefinition>> */
    private array $packRules = [];

    public function registerPack(string $packId, array $ruleDefinitions): void
    {
        $rules = [];
        foreach ($ruleDefinitions as $ruleDef) {
            if (!is_array($ruleDef) || !isset($ruleDef['id'])) {
                continue;
            }

            $scope = $this->normalizeScope($ruleDef);
            $priority = (int) ($ruleDef['priority'] ?? 50);
            $triggers = $this->normalizeTriggers($ruleDef);
            $conditions = $this->normalizeConditions($ruleDef);
            $actions = $this->normalizeActions($ruleDef);

            $rules[] = new RuleDefinition(
                (string) $ruleDef['id'],
                $scope,
                $priority,
                $triggers,
                $conditions,
                $actions,
                isset($ruleDef['on_veto']) && is_array($ruleDef['on_veto']) ? $ruleDef['on_veto'] : null,
                'pack:' . $packId,
                isset($ruleDef['explain']) && is_string($ruleDef['explain']) ? (string) $ruleDef['explain'] : null
            );
        }

        $this->packRules[$packId] = $rules;
    }

    public function getPackRules(string $packId): array
    {
        return $this->packRules[$packId] ?? [];
    }

    public function getAllPackRules(): array
    {
        $allRules = [];
        foreach ($this->packRules as $packId => $rules) {
            $allRules = array_merge($allRules, $rules);
        }
        return $allRules;
    }

    public function getEnabledRules(array $enabledPacks): array
    {
        $rules = [];
        foreach ($enabledPacks as $packId) {
            $rules = array_merge($rules, $this->getPackRules($packId));
        }
        return $rules;
    }

    private function normalizeScope(array $ruleDef): string
    {
        $raw = (string) ($ruleDef['scope'] ?? ($ruleDef['strata'] ?? 'module'));
        if ($raw === '' || $raw === 'pack') {
            return 'module';
        }
        return $raw;
    }

    /**
     * @param array<string,mixed> $ruleDef
     * @return list<string>
     */
    private function normalizeTriggers(array $ruleDef): array
    {
        $triggers = [];

        if (isset($ruleDef['triggers']) && is_array($ruleDef['triggers'])) {
            foreach ($ruleDef['triggers'] as $trigger) {
                if (!is_string($trigger)) {
                    continue;
                }
                $triggers[] = $this->stripCommandPrefix($trigger);
            }
            return $triggers;
        }

        $legacy = $ruleDef['trigger'] ?? null;
        if (is_array($legacy) && isset($legacy['on'])) {
            $value = $legacy['on'];
            if (is_string($value) && $value !== '') {
                $triggers[] = $this->stripCommandPrefix($value);
            }
        } elseif (is_string($legacy) && $legacy !== '') {
            $triggers[] = $this->stripCommandPrefix($legacy);
        }

        return $triggers;
    }

    /**
     * @param array<string,mixed> $ruleDef
     * @return list<array<string,mixed>>
     */
    private function normalizeConditions(array $ruleDef): array
    {
        $conditions = $ruleDef['conditions'] ?? null;
        if (!is_array($conditions)) {
            $conditions = [];
        }

        // Legacy support for "when"
        if ($conditions === [] && isset($ruleDef['when']) && is_array($ruleDef['when'])) {
            $conditions = $this->convertLegacyConditions($ruleDef['when']);
        }

        $normalized = [];
        foreach ($conditions as $condition) {
            if (!is_array($condition)) {
                continue;
            }
            $normalized[] = $condition;
        }

        return $normalized;
    }

    /**
     * @param array<string,mixed> $ruleDef
     * @return list<array<string,mixed>>
     */
    private function normalizeActions(array $ruleDef): array
    {
        $actions = $ruleDef['actions'] ?? null;
        if (!is_array($actions)) {
            $actions = [];
        }

        if ($actions === [] && isset($ruleDef['then']) && is_array($ruleDef['then'])) {
            $actions = $ruleDef['then'];
        }

        $normalized = [];
        foreach ($actions as $action) {
            if (!is_array($action)) {
                continue;
            }

            if (!isset($action['type']) && isset($action['op']) && is_string($action['op'])) {
                $action['type'] = $action['op'];
            }
            if (!isset($action['type']) || !is_string($action['type']) || $action['type'] === '') {
                continue;
            }

            if (isset($action['args']) && is_array($action['args'])) {
                $action = array_merge($action, $action['args']);
                unset($action['args']);
            }

            unset($action['op']);
            $normalized[] = $action;
        }

        return $normalized;
    }

    /**
     * @param list<array<string,mixed>> $conditions
     * @return list<array<string,mixed>>
     */
    private function convertLegacyConditions(array $conditions): array
    {
        $normalized = [];
        foreach ($conditions as $condition) {
            if (!is_array($condition) || !isset($condition['op'])) {
                continue;
            }
            $op = (string) $condition['op'];
            $args = is_array($condition['args'] ?? null) ? $condition['args'] : [];

            if ($op === 'itemHasTag' && isset($args['key'])) {
                $normalized[] = [
                    'when' => 'operation.tag.key',
                    'equals' => (string) $args['key'],
                ];
                continue;
            }

            // Fallback: keep original structure for unknown ops
            $normalized[] = $condition;
        }

        return $normalized;
    }

    private function stripCommandPrefix(string $trigger): string
    {
        if (str_starts_with($trigger, 'command:')) {
            return substr($trigger, 8);
        }

        return $trigger;
    }
}
