<?php

declare(strict_types=1);

namespace Skyboard\Infrastructure\Persistence;

use PDO;
use Skyboard\Application\Services\BoardRepository;
use Skyboard\Domain\Boards\BoardConflictException;
use Skyboard\Domain\Boards\BoardInvariantChecker;
use Skyboard\Domain\Boards\BoardPatch;
use Skyboard\Domain\Boards\BoardSnapshot;
use Skyboard\Domain\Boards\BoardState;
use Skyboard\Domain\Boards\BoardStateApplier;
use Skyboard\Domain\Boards\BoardWriteResult;
use Throwable;

final class MySqlBoardRepository implements BoardRepository
{
    private const HISTORY_LIMIT = 5;

    private PDO $pdo;
    private BoardStateApplier $applier;
    private BoardInvariantChecker $invariants;

    public function __construct(DatabaseConnection $connection)
    {
        $this->pdo = $connection->pdo();
        $this->applier = new BoardStateApplier();
        $this->invariants = new BoardInvariantChecker();
    }

    public function getById(string $boardId, int $userId): ?BoardState
    {
        return $this->getSnapshot($boardId, $userId)?->state();
    }

    public function getSnapshot(string $boardId, int $userId): ?BoardSnapshot
    {
        $stmt = $this->pdo->prepare('SELECT state_json, revision, updated_at, history_json FROM boards WHERE id = :id AND user_id = :user');
        $stmt->execute([
            'id' => $boardId,
            'user' => $userId,
        ]);
        $row = $stmt->fetch();
        if (!$row) {
            return null;
        }
        $state = json_decode((string) $row['state_json'], true);
        if (!is_array($state)) {
            return null;
        }
        if ((int)($state['version'] ?? 0) < 4) {
            throw new \RuntimeException('UPGRADE_REQUIRED: snapshot version < 4');
        }
        $boardState = BoardState::fromArray($state);
        $revision = isset($row['revision']) ? (int) $row['revision'] : (isset($row['updated_at']) ? (int) $row['updated_at'] : 0);
        $updatedAt = isset($row['updated_at']) ? (int) $row['updated_at'] : $revision;
        $history = $this->decodeHistory($row['history_json'] ?? null);
        if ($history === [] && $revision > 0) {
            $history = $this->pushHistory([], $revision, $updatedAt);
        }

        return new BoardSnapshot($boardState, $revision, $updatedAt, $history);
    }

    public function applyPatch(BoardState $board, BoardPatch $patch, int $userId, ?int $expectedRevision = null): BoardWriteResult
    {
        $board = $this->applier->apply($board, $patch);
        $this->invariants->assert($board);
        return $this->persist($patch->boardId(), $userId, $board, $expectedRevision);
    }

    public function overwrite(string $boardId, int $userId, BoardState $state, ?int $expectedRevision = null): BoardWriteResult
    {
        $this->invariants->assert($state);
        return $this->persist($boardId, $userId, $state, $expectedRevision);
    }

    private function persist(string $boardId, int $userId, BoardState $board, ?int $expectedRevision = null): BoardWriteResult
    {
        $payload = json_encode($board->toArray(), JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
        if ($payload === false) {
            throw new \RuntimeException('BOARD_STATE_ENCODING_FAILED');
        }

        $this->pdo->beginTransaction();
        try {
            $metaStmt = $this->pdo->prepare('SELECT revision, updated_at, history_json FROM boards WHERE id = :id AND user_id = :user');
            $metaStmt->execute([
                'id' => $boardId,
                'user' => $userId,
            ]);
            $meta = $metaStmt->fetch();
            if (!$meta) {
                throw new \RuntimeException('BOARD_NOT_FOUND');
            }

            $currentRevision = isset($meta['revision']) ? (int) $meta['revision'] : (isset($meta['updated_at']) ? (int) $meta['updated_at'] : 0);
            if ($expectedRevision !== null && $currentRevision !== $expectedRevision) {
                throw new BoardConflictException('BOARD_CONCURRENT_MODIFICATION');
            }

            $ts = $this->nextTimestamp();
            $existingHistory = $this->decodeHistory($meta['history_json'] ?? null);
            if ($existingHistory === [] && $currentRevision > 0) {
                $existingHistory = $this->pushHistory([], $currentRevision, isset($meta['updated_at']) ? (int) $meta['updated_at'] : $currentRevision);
            }
            $history = $this->pushHistory($existingHistory, $ts, $ts);

            $historyJson = json_encode($history, JSON_UNESCAPED_SLASHES);
            if ($historyJson === false) {
                throw new \RuntimeException('BOARD_HISTORY_ENCODING_FAILED');
            }

            $sql = 'UPDATE boards SET state_json = :state, updated_at = :updatedAt, revision = :revision, history_json = :history WHERE id = :id AND user_id = :user';
            $params = [
                'state' => $payload,
                'updatedAt' => $ts,
                'revision' => $ts,
                'history' => $historyJson,
                'id' => $boardId,
                'user' => $userId,
            ];
            if ($expectedRevision !== null) {
                $sql .= ' AND revision = :expectedRevision';
                $params['expectedRevision'] = $expectedRevision;
            }

            $stmt = $this->pdo->prepare($sql);
            $stmt->execute($params);
            if ($stmt->rowCount() === 0) {
                if ($expectedRevision !== null) {
                    throw new BoardConflictException('BOARD_CONCURRENT_MODIFICATION');
                }
                throw new \RuntimeException('BOARD_NOT_FOUND');
            }

            $this->pdo->commit();

            return new BoardWriteResult($board, $ts, $ts, $history);
        } catch (Throwable $e) {
            if ($this->pdo->inTransaction()) {
                $this->pdo->rollBack();
            }
            throw $e;
        }
    }

    private function nextTimestamp(): int
    {
        // Policy: seconds everywhere in model/APIs
        return time();
    }

    /**
     * @return list<array{revision:int,updatedAt:int}>
     */
    private function decodeHistory(?string $json): array
    {
        if ($json === null || $json === '') {
            return [];
        }

        $decoded = json_decode($json, true);
        if (!is_array($decoded)) {
            return [];
        }

        $history = [];
        foreach ($decoded as $entry) {
            if (!is_array($entry)) {
                continue;
            }
            $revision = isset($entry['revision']) ? (int) $entry['revision'] : 0;
            if ($revision <= 0) {
                continue;
            }
            $updatedAt = isset($entry['updatedAt']) ? (int) $entry['updatedAt'] : $revision;
            $history[] = [
                'revision' => $revision,
                'updatedAt' => $updatedAt,
            ];
            if (count($history) >= self::HISTORY_LIMIT) {
                break;
            }
        }

        return $history;
    }

    /**
     * @param list<array{revision:int,updatedAt:int}> $history
     * @return list<array{revision:int,updatedAt:int}>
     */
    private function pushHistory(array $history, int $revision, int $updatedAt): array
    {
        array_unshift($history, ['revision' => $revision, 'updatedAt' => $updatedAt]);
        $unique = [];
        $result = [];
        foreach ($history as $entry) {
            $rev = isset($entry['revision']) ? (int) $entry['revision'] : 0;
            if ($rev <= 0) {
                continue;
            }
            if (isset($unique[$rev])) {
                continue;
            }
            $unique[$rev] = true;
            $result[] = [
                'revision' => $rev,
                'updatedAt' => isset($entry['updatedAt']) ? (int) $entry['updatedAt'] : $rev,
            ];
            if (count($result) >= self::HISTORY_LIMIT) {
                break;
            }
        }

        return $result;
    }
}
