<?php

declare(strict_types=1);

namespace Skyboard\Infrastructure\Persistence;

use PDO;
use Skyboard\Application\Services\UserFileFolderRepository;
use Skyboard\Domain\UserFiles\UserFileFolder;

final class MySqlUserFileFolderRepository implements UserFileFolderRepository
{
    public function __construct(private readonly DatabaseConnection $connection)
    {
    }

    private function pdo(): PDO { return $this->connection->pdo(); }

    /**
     * @return list<UserFileFolder>
     */
    public function listForUser(int $userId): array
    {
        $stmt = $this->pdo()->prepare('SELECT * FROM user_file_folders WHERE user_id = :user ORDER BY name ASC, id ASC');
        $stmt->execute(['user' => $userId]);
        $rows = $stmt->fetchAll(PDO::FETCH_ASSOC) ?: [];
        $out = [];
        foreach ($rows as $row) {
            $mapped = $this->mapRow($row);
            if ($mapped) $out[] = $mapped;
        }
        return $out;
    }

    public function insert(UserFileFolder $folder): UserFileFolder
    {
        $stmt = $this->pdo()->prepare('INSERT INTO user_file_folders (user_id, public_id, name, parent_id, created_at, updated_at)
            VALUES (:user_id, :public_id, :name, :parent_id, :created_at, :updated_at)');
        $rec = $folder->toRecord();
        $stmt->execute([
            'user_id' => $rec['user_id'],
            'public_id' => $rec['public_id'],
            'name' => $rec['name'],
            'parent_id' => $rec['parent_id'],
            'created_at' => $rec['created_at'],
            'updated_at' => $rec['updated_at'],
        ]);
        $id = (int) $this->pdo()->lastInsertId();
        return $folder->withId($id);
    }

    public function findByPublicId(int $userId, string $publicId): ?UserFileFolder
    {
        $stmt = $this->pdo()->prepare('SELECT * FROM user_file_folders WHERE user_id = :user AND public_id = :public LIMIT 1');
        $stmt->execute(['user' => $userId, 'public' => $publicId]);
        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        if (!$row || !is_array($row)) return null;
        return $this->mapRow($row);
    }

    public function update(UserFileFolder $folder): void
    {
        $rec = $folder->toRecord();
        $stmt = $this->pdo()->prepare('UPDATE user_file_folders
            SET name = :name, parent_id = :parent_id, updated_at = :updated_at
            WHERE user_id = :user_id AND public_id = :public_id');
        $stmt->execute([
            'name' => $rec['name'],
            'parent_id' => $rec['parent_id'],
            'updated_at' => $rec['updated_at'],
            'user_id' => $rec['user_id'],
            'public_id' => $rec['public_id'],
        ]);
    }

    public function delete(UserFileFolder $folder): void
    {
        $stmt = $this->pdo()->prepare('DELETE FROM user_file_folders WHERE user_id = :user AND public_id = :public');
        $stmt->execute(['user' => $folder->userId(), 'public' => $folder->publicId()]);
    }

    public function countFilesInFolder(int $userId, int $folderId): int
    {
        $stmt = $this->pdo()->prepare('SELECT COUNT(*) FROM user_files WHERE user_id = :user AND folder_id = :folder');
        $stmt->execute(['user' => $userId, 'folder' => $folderId]);
        $n = $stmt->fetchColumn();
        return is_numeric($n) ? (int) $n : 0;
    }

    public function countSubFolders(int $userId, int $parentId): int
    {
        $stmt = $this->pdo()->prepare('SELECT COUNT(*) FROM user_file_folders WHERE user_id = :user AND parent_id = :parent');
        $stmt->execute(['user' => $userId, 'parent' => $parentId]);
        $n = $stmt->fetchColumn();
        return is_numeric($n) ? (int) $n : 0;
    }

    /**
     * @param array<string,mixed> $row
     */
    private function mapRow(array $row): ?UserFileFolder
    {
        try {
            return new UserFileFolder(
                isset($row['id']) ? (int) $row['id'] : null,
                isset($row['user_id']) ? (int) $row['user_id'] : 0,
                (string) ($row['public_id'] ?? ''),
                (string) ($row['name'] ?? ''),
                isset($row['parent_id']) ? (int) $row['parent_id'] : null,
                isset($row['created_at']) ? (int) $row['created_at'] : time(),
                isset($row['updated_at']) ? (int) $row['updated_at'] : time(),
            );
        } catch (\Throwable) {
            return null;
        }
    }
}
