<?php

declare(strict_types=1);

namespace Skyboard\Infrastructure\Persistence;

use PDO;
use Throwable;

final class SchemaManager
{
    public function __construct(private readonly DatabaseConnection $connection)
    {
    }

    public function migrate(): void
    {
        $pdo = $this->connection->pdo();
        $driver = $this->connection->driver();
        if ($driver !== 'mysql') {
            throw new \RuntimeException('SCHEMA_UNSUPPORTED_DRIVER: seul MySQL est supporté.');
        }

        $this->migrateMySql($pdo);
    }

    private function migrateMySql(PDO $pdo): void
    {
        $commonOptions = ' ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci';

        $pdo->exec(<<<'SQL'
CREATE TABLE IF NOT EXISTS users (
    id INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    email VARCHAR(255) NOT NULL UNIQUE,
    password_hash VARCHAR(255) NOT NULL,
    created_at BIGINT UNSIGNED NOT NULL,
    email_verified_at BIGINT UNSIGNED NULL,
    verify_token VARCHAR(255) NULL,
    verify_expires_at BIGINT UNSIGNED NULL,
    blocked TINYINT(1) NOT NULL DEFAULT 0
)
SQL
            . $commonOptions);

        $pdo->exec(<<<'SQL'
CREATE TABLE IF NOT EXISTS user_profiles (
    user_id INT UNSIGNED PRIMARY KEY,
    pseudo VARCHAR(255) NULL,
    role VARCHAR(64) NOT NULL DEFAULT 'standard',
    CONSTRAINT fk_user_profiles_user FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE
)
SQL
            . $commonOptions);

        // Folders for user files (hierarchical)
        $pdo->exec(<<<'SQL'
CREATE TABLE IF NOT EXISTS user_file_folders (
    id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    user_id INT UNSIGNED NOT NULL,
    public_id VARCHAR(191) NOT NULL UNIQUE,
    name VARCHAR(255) NOT NULL,
    parent_id BIGINT UNSIGNED NULL,
    created_at BIGINT UNSIGNED NOT NULL,
    updated_at BIGINT UNSIGNED NOT NULL,
    CONSTRAINT fk_user_file_folders_user FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    CONSTRAINT fk_user_file_folders_parent FOREIGN KEY (parent_id) REFERENCES user_file_folders(id) ON DELETE SET NULL
)
SQL
            . $commonOptions);

        $pdo->exec(<<<'SQL'
CREATE TABLE IF NOT EXISTS user_files (
    id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    user_id INT UNSIGNED NOT NULL,
    public_id VARCHAR(191) NOT NULL UNIQUE,
    stored_name VARCHAR(255) NOT NULL,
    original_name VARCHAR(255) NOT NULL,
    mime_type VARCHAR(191) NULL,
    byte_size BIGINT UNSIGNED NOT NULL,
    checksum VARCHAR(191) NULL,
    folder_id BIGINT UNSIGNED NULL,
    created_at BIGINT UNSIGNED NOT NULL,
    updated_at BIGINT UNSIGNED NOT NULL,
    CONSTRAINT fk_user_files_user FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    CONSTRAINT fk_user_files_folder FOREIGN KEY (folder_id) REFERENCES user_file_folders(id) ON DELETE SET NULL
)
SQL
            . $commonOptions);

        // Indexes
        try {
            $this->createIndex($pdo, 'CREATE INDEX idx_user_files_user ON user_files(user_id)');
        } catch (Throwable) {}
        try {
            $this->createIndex($pdo, 'CREATE INDEX idx_user_files_folder ON user_files(folder_id)');
        } catch (Throwable) {}
        try {
            $this->createIndex($pdo, 'CREATE INDEX idx_user_file_folders_user ON user_file_folders(user_id)');
        } catch (Throwable) {}
        try {
            $this->createIndex($pdo, 'CREATE INDEX idx_user_file_folders_parent ON user_file_folders(parent_id)');
        } catch (Throwable) {}

        // Backfill/upgrade for existing installs: ensure user_files.folder_id exists + FK
        $this->ensureUserFilesFolderColumn($pdo);
        // Backfill/upgrade: ensure boards.thumbnail_public_id exists
        $this->ensureBoardsThumbnailColumn($pdo);

        $pdo->exec(<<<'SQL'
CREATE TABLE IF NOT EXISTS boards (
    id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    user_id INT UNSIGNED NOT NULL,
    title VARCHAR(255) NOT NULL,
    state_json LONGTEXT NOT NULL,
    updated_at BIGINT UNSIGNED NOT NULL,
    created_at BIGINT UNSIGNED NOT NULL,
    revision BIGINT UNSIGNED NOT NULL DEFAULT 0,
    history_json LONGTEXT NULL,
    CONSTRAINT fk_boards_user FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE
)
SQL
            . $commonOptions);

        $pdo->exec(<<<'SQL'
CREATE TABLE IF NOT EXISTS board_rules (
    id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    board_id BIGINT UNSIGNED NOT NULL,
    scope VARCHAR(100) NOT NULL,
    priority INT NOT NULL DEFAULT 0,
    triggers_json LONGTEXT NOT NULL,
    conditions_json LONGTEXT NOT NULL,
    actions_json LONGTEXT NOT NULL,
    on_veto_json LONGTEXT NULL,
    created_at BIGINT UNSIGNED NOT NULL,
    CONSTRAINT fk_board_rules_board FOREIGN KEY (board_id) REFERENCES boards(id) ON DELETE CASCADE
)
SQL
            . $commonOptions);

        $pdo->exec(<<<'SQL'
CREATE TABLE IF NOT EXISTS licenses (
    id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    code VARCHAR(191) NOT NULL UNIQUE,
    role VARCHAR(100) NOT NULL,
    packs_json LONGTEXT NOT NULL,
    expires_at BIGINT UNSIGNED NULL,
    assigned_user_id INT UNSIGNED NULL,
    redeemed_at BIGINT UNSIGNED NULL,
    created_at BIGINT UNSIGNED NOT NULL,
    CONSTRAINT fk_licenses_assigned_user FOREIGN KEY (assigned_user_id) REFERENCES users(id) ON DELETE SET NULL
)
SQL
            . $commonOptions);

        $pdo->exec(<<<'SQL'
CREATE TABLE IF NOT EXISTS user_licenses (
    user_id INT UNSIGNED NOT NULL,
    license_id BIGINT UNSIGNED NOT NULL,
    PRIMARY KEY(user_id, license_id),
    CONSTRAINT fk_user_licenses_user FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    CONSTRAINT fk_user_licenses_license FOREIGN KEY (license_id) REFERENCES licenses(id) ON DELETE CASCADE
)
SQL
            . $commonOptions);

        $pdo->exec(<<<'SQL'
CREATE TABLE IF NOT EXISTS license_links (
    id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    user_id INT UNSIGNED NOT NULL,
    token VARCHAR(191) NOT NULL UNIQUE,
    code VARCHAR(191) NOT NULL,
    expires_at BIGINT UNSIGNED NOT NULL,
    used_at BIGINT UNSIGNED NULL,
    created_at BIGINT UNSIGNED NOT NULL,
    CONSTRAINT fk_license_links_user FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE
)
SQL
            . $commonOptions);

        // legacy notifications tables removed (replaced by notifications_rich + notification_user_state)

        $pdo->exec(<<<'SQL'
CREATE TABLE IF NOT EXISTS ip_rec (
    ip VARCHAR(64) PRIMARY KEY,
    fails INT UNSIGNED NOT NULL DEFAULT 0,
    banned_until BIGINT UNSIGNED NOT NULL DEFAULT 0
)
SQL
            . $commonOptions);

        $pdo->exec(<<<'SQL'
CREATE TABLE IF NOT EXISTS sessions (
    token VARCHAR(191) PRIMARY KEY,
    user_id INT UNSIGNED NOT NULL,
    csrf_token VARCHAR(191) NOT NULL,
    created_at BIGINT UNSIGNED NOT NULL,
    last_seen BIGINT UNSIGNED NOT NULL,
    CONSTRAINT fk_sessions_user FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE
)
SQL
            . $commonOptions);

        $this->createIndex($pdo, 'CREATE INDEX idx_boards_user ON boards(user_id)');
        // legacy idx_notifications_active removed
        $this->createIndex($pdo, 'CREATE INDEX idx_user_files_user ON user_files(user_id)');

        $pdo->exec('UPDATE boards SET revision = updated_at WHERE revision IS NULL OR revision = 0');
        $pdo->exec("UPDATE boards SET history_json = '[]' WHERE history_json IS NULL");

        // --- V4: Activity domain (append-only) + user state pivot
        $pdo->exec(<<<'SQL'
CREATE TABLE IF NOT EXISTS activity_events (
    id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    board_id BIGINT UNSIGNED NOT NULL,
    actor_id BIGINT UNSIGNED NULL,
    kind VARCHAR(64) NOT NULL,
    payload_json JSON NOT NULL,
    severity TINYINT UNSIGNED NOT NULL DEFAULT 0,
    created_at BIGINT UNSIGNED NOT NULL
)
SQL
            . $commonOptions);

        $pdo->exec(<<<'SQL'
CREATE TABLE IF NOT EXISTS activity_user_state (
    user_id INT UNSIGNED NOT NULL,
    event_id BIGINT UNSIGNED NOT NULL,
    seen_at BIGINT UNSIGNED NULL,
    archived_at BIGINT UNSIGNED NULL,
    deleted_at BIGINT UNSIGNED NULL,
    status ENUM('UNREAD','SEEN','ARCHIVED','DELETED')
        GENERATED ALWAYS AS (
            CASE
                WHEN deleted_at  IS NOT NULL THEN 'DELETED'
                WHEN archived_at IS NOT NULL THEN 'ARCHIVED'
                WHEN seen_at     IS NOT NULL THEN 'SEEN'
                ELSE 'UNREAD'
            END
        ) STORED,
    PRIMARY KEY(user_id, event_id),
    CONSTRAINT fk_activity_state_event FOREIGN KEY (event_id) REFERENCES activity_events(id) ON DELETE CASCADE
)
SQL
            . $commonOptions);

        // Indexes for activity
        $this->createIndex($pdo, 'CREATE INDEX idx_activity_board_date ON activity_events(board_id, created_at)');
        $this->createIndex($pdo, 'CREATE INDEX idx_activity_kind_date ON activity_events(kind, created_at)');
        $this->createIndex($pdo, 'CREATE INDEX ix_aus_user_status ON activity_user_state(user_id, status)');
        $this->createIndex($pdo, 'CREATE INDEX ix_aus_user_seen ON activity_user_state(user_id, seen_at)');

        // --- V4: Rich notifications (editorial) + categories + user state pivot
        $pdo->exec(<<<'SQL'
CREATE TABLE IF NOT EXISTS notification_categories (
    id INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    slug VARCHAR(64) NOT NULL UNIQUE,
    name VARCHAR(128) NOT NULL,
    sort_order INT NOT NULL DEFAULT 0,
    active TINYINT(1) NOT NULL DEFAULT 1
)
SQL
            . $commonOptions);

        $pdo->exec(<<<'SQL'
CREATE TABLE IF NOT EXISTS notifications_rich (
    id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    category_id INT UNSIGNED NOT NULL,
    title VARCHAR(255) NOT NULL,
    emitter VARCHAR(128) NULL,
    weight INT NOT NULL DEFAULT 0,
    active TINYINT(1) NOT NULL DEFAULT 1,
    content_html LONGTEXT NOT NULL,
    content_css LONGTEXT NULL,
    created_at BIGINT UNSIGNED NOT NULL,
    CONSTRAINT fk_notifications_rich_category FOREIGN KEY (category_id) REFERENCES notification_categories(id)
)
SQL
            . $commonOptions);

        $this->createIndex($pdo, 'CREATE INDEX idx_notif_cat_active_date ON notifications_rich(category_id, active, created_at)');
        $this->createIndex($pdo, 'CREATE INDEX idx_notif_weight_date ON notifications_rich(weight, created_at)');
        // Drop legacy content_js column if it exists (strict V4: no author JS)
        try { $pdo->exec("ALTER TABLE notifications_rich DROP COLUMN content_js"); } catch (Throwable) {}

        $pdo->exec(<<<'SQL'
CREATE TABLE IF NOT EXISTS notification_user_state (
    user_id INT UNSIGNED NOT NULL,
    notification_id BIGINT UNSIGNED NOT NULL,
    seen_at BIGINT UNSIGNED NULL,
    archived_at BIGINT UNSIGNED NULL,
    deleted_at BIGINT UNSIGNED NULL,
    pinned_at BIGINT UNSIGNED NULL,
    muted_until BIGINT UNSIGNED NULL,
    last_opened_at BIGINT UNSIGNED NULL,
    status ENUM('UNREAD','SEEN','ARCHIVED','DELETED')
        GENERATED ALWAYS AS (
            CASE
                WHEN deleted_at  IS NOT NULL THEN 'DELETED'
                WHEN archived_at IS NOT NULL THEN 'ARCHIVED'
                WHEN seen_at     IS NOT NULL THEN 'SEEN'
                ELSE 'UNREAD'
            END
        ) STORED,
    PRIMARY KEY(user_id, notification_id),
    CONSTRAINT fk_notification_state_notification FOREIGN KEY (notification_id) REFERENCES notifications_rich(id) ON DELETE CASCADE
)
SQL
            . $commonOptions);

        $this->createIndex($pdo, 'CREATE INDEX ix_nus_user_status ON notification_user_state(user_id, status)');
        $this->createIndex($pdo, 'CREATE INDEX ix_nus_user_unread ON notification_user_state(user_id, deleted_at, archived_at, seen_at)');

        // --- Alter/extend tables for subscriptions & scheduling (Phase 1)
        try { $pdo->exec("ALTER TABLE notification_categories ADD COLUMN audience_mode ENUM('EVERYONE','SUBSCRIBERS') NOT NULL DEFAULT 'EVERYONE'"); } catch (Throwable) {}
        try { $pdo->exec("ALTER TABLE notification_categories ADD COLUMN dispatch_mode ENUM('BROADCAST','PERSONALIZED') NOT NULL DEFAULT 'BROADCAST'"); } catch (Throwable) {}
        try { $pdo->exec("ALTER TABLE notification_categories ADD COLUMN frequency_kind ENUM('IMMEDIATE','EVERY_N_DAYS','WEEKLY','MONTHLY') NOT NULL DEFAULT 'IMMEDIATE'"); } catch (Throwable) {}
        try { $pdo->exec("ALTER TABLE notification_categories ADD COLUMN frequency_param TINYINT UNSIGNED NULL"); } catch (Throwable) {}
        try { $pdo->exec("ALTER TABLE notification_categories ADD COLUMN anchor_ts BIGINT UNSIGNED NULL"); } catch (Throwable) {}
        try { $pdo->exec("ALTER TABLE notification_categories ADD COLUMN allow_user_override TINYINT(1) NOT NULL DEFAULT 0"); } catch (Throwable) {}

        try { $pdo->exec('ALTER TABLE notifications_rich ADD COLUMN sequence_index INT UNSIGNED NOT NULL'); } catch (Throwable) {}
        try { $pdo->exec('ALTER TABLE notifications_rich ADD UNIQUE KEY uq_notif_sequence (category_id, sequence_index)'); } catch (Throwable) {}
        // Actions catalogue integration (V1): references + snapshot-light per notification
        try { $pdo->exec("ALTER TABLE notifications_rich ADD COLUMN actions_ref_json JSON NOT NULL"); } catch (Throwable) {}
        try { $pdo->exec("ALTER TABLE notifications_rich ADD COLUMN actions_snapshot_json JSON NOT NULL"); } catch (Throwable) {}

        try { $pdo->exec('ALTER TABLE notification_user_state ADD COLUMN available_at BIGINT UNSIGNED NULL'); } catch (Throwable) {}
        try { $pdo->exec('ALTER TABLE notification_user_state ADD COLUMN delivered_at BIGINT UNSIGNED NULL'); } catch (Throwable) {}
        $this->createIndex($pdo, 'CREATE INDEX ix_nus_user_available ON notification_user_state(user_id, available_at)');
        // delivered_at index to be enabled when transport enabled (email/push)
        // $this->createIndex($pdo, 'CREATE INDEX ix_nus_user_delivered ON notification_user_state(user_id, delivered_at)');

        // Subscriptions table
        $pdo->exec(<<<'SQL'
CREATE TABLE IF NOT EXISTS user_subscriptions (
  user_id INT UNSIGNED NOT NULL,
  category_id INT UNSIGNED NOT NULL,
  subscribed TINYINT(1) NOT NULL DEFAULT 1,
  override_kind ENUM('IMMEDIATE','EVERY_N_DAYS','WEEKLY','MONTHLY') NULL,
  override_param TINYINT UNSIGNED NULL,
  cycle_anchor_ts BIGINT UNSIGNED NULL,
  last_delivered_notification_id BIGINT UNSIGNED NULL,
  created_at BIGINT UNSIGNED NOT NULL,
  PRIMARY KEY (user_id, category_id),
  KEY ix_us_cat (category_id),
  CONSTRAINT fk_us_user FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
  CONSTRAINT fk_us_cat FOREIGN KEY (category_id) REFERENCES notification_categories(id) ON DELETE CASCADE
)
SQL
            . $commonOptions);

        // Performance index for broadcast deliveries by category + subscription flag
        $this->createIndex($pdo, 'CREATE INDEX ix_us_cat_sub ON user_subscriptions(category_id, subscribed)');

        // Generic per-user KV store for future features
        $pdo->exec(<<<'SQL'
CREATE TABLE IF NOT EXISTS user_data (
  user_id INT UNSIGNED NOT NULL,
  namespace VARCHAR(64) NOT NULL,
  `key` VARCHAR(128) NOT NULL,
  value_json JSON NOT NULL,
  updated_at BIGINT UNSIGNED NOT NULL,
  PRIMARY KEY (user_id, namespace, `key`),
  CONSTRAINT fk_ud_user FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE
)
SQL
            . $commonOptions);

        // Achievements (optional)
        $pdo->exec(<<<'SQL'
CREATE TABLE IF NOT EXISTS user_achievements (
  user_id INT UNSIGNED NOT NULL,
  code VARCHAR(64) NOT NULL,
  earned_at BIGINT UNSIGNED NOT NULL,
  metadata JSON NULL,
  PRIMARY KEY (user_id, code),
  CONSTRAINT fk_ua_user FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE
)
SQL
            . $commonOptions);

        // Scheduler cursor per category (BROADCAST)
        $pdo->exec(<<<'SQL'
CREATE TABLE IF NOT EXISTS scheduler_meta (
  category_id INT UNSIGNED NOT NULL,
  last_slot_ts BIGINT UNSIGNED NOT NULL DEFAULT 0,
  PRIMARY KEY (category_id),
  CONSTRAINT fk_sm_cat FOREIGN KEY (category_id) REFERENCES notification_categories(id) ON DELETE CASCADE
)
SQL
            . $commonOptions);

        // --- V4: Actions catalogue (versioned)
        $pdo->exec(<<<'SQL'
CREATE TABLE IF NOT EXISTS action_catalog (
  action_id VARCHAR(128) NOT NULL PRIMARY KEY,
  kind VARCHAR(128) NOT NULL,
  created_at BIGINT UNSIGNED NOT NULL,
  created_by INT UNSIGNED NULL
)
SQL
            . $commonOptions);

        $pdo->exec(<<<'SQL'
CREATE TABLE IF NOT EXISTS action_catalog_versions (
  action_id VARCHAR(128) NOT NULL,
  version INT NOT NULL,
  status ENUM('draft','active','deprecated','disabled') NOT NULL DEFAULT 'draft',
  definition_json JSON NOT NULL,
  etag VARCHAR(80) NOT NULL,
  updated_at BIGINT UNSIGNED NOT NULL,
  updated_by INT UNSIGNED NULL,
  PRIMARY KEY (action_id, version),
  CONSTRAINT fk_acv_action FOREIGN KEY (action_id) REFERENCES action_catalog(action_id) ON DELETE CASCADE
)
SQL
            . $commonOptions);

        // Indexes for catalogue lookups
        $this->createIndex($pdo, 'CREATE INDEX idx_acv_status ON action_catalog_versions(status)');
        $this->createIndex($pdo, 'CREATE INDEX idx_acv_action_status ON action_catalog_versions(action_id, status)');
        // Optional uniqueness to ensure coherence between version and etag
        try { $pdo->exec('CREATE UNIQUE INDEX uq_acv_action_version_etag ON action_catalog_versions(action_id, version, etag)'); } catch (Throwable) {}
    }

    private function createIndex(PDO $pdo, string $sql): void
    {
        try {
            $pdo->exec($sql);
        } catch (Throwable) {
        }
    }

    private function ensureUserFilesFolderColumn(PDO $pdo): void
    {
        try {
            $exists = false;
            $stmt = $pdo->query("SHOW COLUMNS FROM user_files LIKE 'folder_id'");
            if ($stmt !== false) {
                $row = $stmt->fetch(PDO::FETCH_ASSOC);
                $exists = is_array($row);
            }
            if (!$exists) {
                try { $pdo->exec('ALTER TABLE user_files ADD COLUMN folder_id BIGINT UNSIGNED NULL'); } catch (Throwable) {}
            }
            // Try adding FK (ignore if already there)
            try { $pdo->exec('ALTER TABLE user_files ADD CONSTRAINT fk_user_files_folder FOREIGN KEY (folder_id) REFERENCES user_file_folders(id) ON DELETE SET NULL'); } catch (Throwable) {}
        } catch (Throwable) {
            // Silent: we don't want to block boot if privileges are limited
        }
    }

    private function ensureBoardsThumbnailColumn(PDO $pdo): void
    {
        try {
            $exists = false;
            $stmt = $pdo->query("SHOW COLUMNS FROM boards LIKE 'thumbnail_public_id'");
            if ($stmt !== false) {
                $row = $stmt->fetch(PDO::FETCH_ASSOC);
                $exists = is_array($row);
            }
            if (!$exists) {
                try { $pdo->exec('ALTER TABLE boards ADD COLUMN thumbnail_public_id VARCHAR(191) NULL'); } catch (Throwable) {}
            }
        } catch (Throwable) {
            // Silent to avoid blocking boot on limited privileges
        }
    }
}
