<?php

declare(strict_types=1);

namespace Skyboard\Interfaces\Http\Controllers\Admin;

use Skyboard\Application\Services\Admin\ActionCatalogAdminService;
use Skyboard\Infrastructure\Http\Request;
use Skyboard\Infrastructure\Http\Response;

final class ActionCatalogAdminController
{
    use AdminControllerTrait;

    public function __construct(private readonly ActionCatalogAdminService $service)
    {
    }

    public function list(Request $request): Response
    {
        if ($r = $this->requireAdmin($request)) return $r;
        return Response::ok(['actions' => $this->service->list()]);
    }

    public function get(Request $request): Response
    {
        if ($r = $this->requireAdmin($request)) return $r;
        $raw = $request->getAttribute('id');
        $id = is_string($raw) ? $raw : '';
        if ($id === '') return Response::error('INVALID_PAYLOAD', 'Identifiant invalide.', [], 422);
        try {
            $payload = $this->service->get($id);
            return Response::ok(['action' => $payload]);
        } catch (\RuntimeException $e) {
            return Response::error('NOT_FOUND', 'Action introuvable.', [], 404);
        }
    }

    public function createAction(Request $request): Response
    {
        if ($r = $this->requireAdmin($request)) return $r;
        $body = $request->body ?? [];
        try {
            $payload = $this->service->createAction($body);
            return Response::ok(['action' => $payload], 201);
        } catch (\InvalidArgumentException $e) {
            return Response::error('INVALID_PAYLOAD', $e->getMessage(), [], 422);
        } catch (\Throwable $e) {
            return Response::error('CREATE_FAILED', $e->getMessage(), [], 422);
        }
    }

    public function createVersion(Request $request): Response
    {
        if ($r = $this->requireAdmin($request)) return $r;
        $body = $request->body ?? [];
        try {
            $payload = $this->service->createVersion($body);
            return Response::ok(['action' => $payload], 201);
        } catch (\InvalidArgumentException $e) {
            return Response::error('INVALID_PAYLOAD', $e->getMessage(), [], 422);
        } catch (\RuntimeException $e) {
            return Response::error('NOT_FOUND', $e->getMessage(), [], 404);
        } catch (\Throwable $e) {
            return Response::error('CREATE_FAILED', $e->getMessage(), [], 422);
        }
    }

    public function updateVersionStatus(Request $request): Response
    {
        if ($r = $this->requireAdmin($request)) return $r;
        $body = $request->body ?? [];
        $actionId = (string) ($body['action_id'] ?? '');
        $version = isset($body['version']) ? (int) $body['version'] : 0;
        $status = (string) ($body['status'] ?? '');
        if ($actionId === '' || $version <= 0 || $status === '') {
            return Response::error('INVALID_PAYLOAD', 'Données invalides.', [], 422);
        }
        try {
            $payload = $this->service->updateVersionStatus($actionId, $version, $status, null);
            return Response::ok(['action' => $payload]);
        } catch (\InvalidArgumentException $e) {
            return Response::error('INVALID_PAYLOAD', $e->getMessage(), [], 422);
        } catch (\RuntimeException $e) {
            return Response::error('NOT_FOUND', $e->getMessage(), [], 404);
        } catch (\Throwable $e) {
            return Response::error('UPDATE_FAILED', $e->getMessage(), [], 422);
        }
    }

    public function deleteVersion(Request $request): Response
    {
        if ($r = $this->requireAdmin($request)) return $r;
        $body = $request->body ?? [];
        $actionId = (string) ($body['action_id'] ?? '');
        $version = isset($body['version']) ? (int) $body['version'] : 0;
        if ($actionId === '' || $version <= 0) {
            return Response::error('INVALID_PAYLOAD', 'Données invalides.', [], 422);
        }
        try {
            $payload = $this->service->deleteVersion($actionId, $version);
            return Response::ok(['action' => $payload]);
        } catch (\InvalidArgumentException $e) {
            $code = $e->getMessage() === 'INVALID_OPERATION_ACTIVE_VERSION' ? 'INVALID_OPERATION' : 'INVALID_PAYLOAD';
            return Response::error($code, $e->getMessage(), [], 422);
        } catch (\RuntimeException $e) {
            $msg = $e->getMessage();
            if ($msg === 'ACTION_VERSION_IN_USE') return Response::error('ACTION_VERSION_IN_USE', 'Version référencée par une notification.', [], 409);
            if ($msg === 'NOT_FOUND') return Response::error('NOT_FOUND', 'Version introuvable.', [], 404);
            return Response::error('DELETE_FAILED', $msg, [], 422);
        } catch (\Throwable $e) {
            return Response::error('DELETE_FAILED', $e->getMessage(), [], 422);
        }
    }

    public function deleteAction(Request $request): Response
    {
        if ($r = $this->requireAdmin($request)) return $r;
        $id = (string) ($request->body['action_id'] ?? '');
        if ($id === '') return Response::error('INVALID_PAYLOAD', 'Données invalides.', [], 422);
        try {
            $this->service->deleteAction($id);
            return Response::ok(['deleted' => true]);
        } catch (\RuntimeException $e) {
            $msg = $e->getMessage();
            if ($msg === 'ACTION_IN_USE') return Response::error('ACTION_IN_USE', 'Action référencée par une notification.', [], 409);
            if ($msg === 'NOT_FOUND') return Response::error('NOT_FOUND', 'Action introuvable.', [], 404);
            return Response::error('DELETE_FAILED', $msg, [], 422);
        } catch (\Throwable $e) {
            return Response::error('DELETE_FAILED', $e->getMessage(), [], 422);
        }
    }
}
