<?php

declare(strict_types=1);

namespace Skyboard\Interfaces\Http\Controllers\Admin;

use Skyboard\Application\Services\Admin\LicenseAdminService;
use Skyboard\Infrastructure\Http\Request;
use Skyboard\Infrastructure\Http\Response;

final class LicenseAdminController
{
    use AdminControllerTrait;

    public function __construct(private readonly LicenseAdminService $licenses)
    {
    }

    public function list(Request $request): Response
    {
        if ($response = $this->requireAdmin($request)) {
            return $response;
        }

        return Response::ok(['licenses' => $this->licenses->list()]);
    }

    public function create(Request $request): Response
    {
        if ($response = $this->requireAdmin($request)) {
            return $response;
        }

        try {
            $license = $this->licenses->create($request->body ?? []);
        } catch (\Throwable $e) {
            return Response::error('CREATE_FAILED', $e->getMessage(), [], 422);
        }

        return Response::ok(['license' => $license], 201);
    }

    public function assign(Request $request): Response
    {
        if ($response = $this->requireAdmin($request)) {
            return $response;
        }

        $body = $request->body ?? [];
        $licenseId = isset($body['licenseId']) ? (int) $body['licenseId'] : 0;
        $userId = isset($body['userId']) ? (int) $body['userId'] : 0;
        if ($licenseId <= 0 || $userId <= 0) {
            return Response::error('INVALID_PAYLOAD', 'Données invalides.', [], 422);
        }

        try {
            $license = $this->licenses->assign($licenseId, $userId);
        } catch (\Throwable $e) {
            return Response::error('ASSIGN_FAILED', $e->getMessage(), [], 422);
        }

        return Response::ok(['license' => $license]);
    }

    public function unassign(Request $request): Response
    {
        if ($response = $this->requireAdmin($request)) {
            return $response;
        }

        $licenseId = isset($request->body['licenseId']) ? (int) $request->body['licenseId'] : 0;
        if ($licenseId <= 0) {
            return Response::error('INVALID_PAYLOAD', 'Données invalides.', [], 422);
        }

        try {
            $license = $this->licenses->unassign($licenseId);
        } catch (\Throwable $e) {
            return Response::error('UNASSIGN_FAILED', $e->getMessage(), [], 422);
        }

        return Response::ok(['license' => $license]);
    }

    public function link(Request $request): Response
    {
        if ($response = $this->requireAdmin($request)) {
            return $response;
        }

        $body = $request->body ?? [];
        $licenseId = isset($body['licenseId']) ? (int) $body['licenseId'] : 0;
        $userId = isset($body['userId']) ? (int) $body['userId'] : 0;
        $ttl = isset($body['ttlSeconds']) ? (int) $body['ttlSeconds'] : 86400;

        if ($licenseId <= 0 || $userId <= 0) {
            return Response::error('INVALID_PAYLOAD', 'Données invalides.', [], 422);
        }

        try {
            $link = $this->licenses->generateActivationLink($licenseId, $userId, $ttl);
        } catch (\Throwable $e) {
            return Response::error('LINK_FAILED', $e->getMessage(), [], 422);
        }

        return Response::ok(['link' => $link], 201);
    }

    public function delete(Request $request): Response
    {
        if ($response = $this->requireAdmin($request)) {
            return $response;
        }

        $licenseId = isset($request->body['licenseId']) ? (int) $request->body['licenseId'] : 0;
        if ($licenseId <= 0) {
            return Response::error('INVALID_PAYLOAD', 'Données invalides.', [], 422);
        }

        $this->licenses->delete($licenseId);
        return Response::ok(['deleted' => true]);
    }
}
