<?php

declare(strict_types=1);

namespace Skyboard\Interfaces\Http\Controllers\Admin;

use Skyboard\Application\Services\Admin\NotificationCategoryAdminService;
use Skyboard\Infrastructure\Http\Request;
use Skyboard\Infrastructure\Http\Response;

final class NotificationCategoryAdminController
{
    use AdminControllerTrait;

    public function __construct(private readonly NotificationCategoryAdminService $categories)
    {
    }

    public function list(Request $request): Response
    {
        if ($response = $this->requireAdmin($request)) return $response;
        return Response::ok(['categories' => $this->categories->list()]);
    }

    public function create(Request $request): Response
    {
        if ($response = $this->requireAdmin($request)) return $response;
        try {
            $category = $this->categories->create($request->body ?? []);
        } catch (\Throwable $e) {
            return Response::error('CREATE_FAILED', $e->getMessage(), [], 422);
        }
        return Response::ok(['category' => $category], 201);
    }

    public function update(Request $request): Response
    {
        if ($response = $this->requireAdmin($request)) return $response;
        $body = $request->body ?? [];
        $id = isset($body['id']) ? (int) $body['id'] : 0;
        if ($id <= 0) return Response::error('INVALID_PAYLOAD', 'Données invalides.', [], 422);
        try {
            $category = $this->categories->update($id, $body);
        } catch (\RuntimeException $e) {
            return Response::error('NOT_FOUND', $e->getMessage(), [], 404);
        } catch (\Throwable $e) {
            return Response::error('UPDATE_FAILED', $e->getMessage(), [], 422);
        }
        return Response::ok(['category' => $category]);
    }

    public function delete(Request $request): Response
    {
        if ($response = $this->requireAdmin($request)) return $response;
        $id = isset($request->body['id']) ? (int) $request->body['id'] : 0;
        if ($id <= 0) return Response::error('INVALID_PAYLOAD', 'Données invalides.', [], 422);
        try {
            $report = $this->categories->delete($id);
        } catch (\RuntimeException $e) {
            if ($e->getMessage() === 'CATEGORY_NOT_FOUND') {
                return Response::error('NOT_FOUND', 'Catégorie introuvable.', [], 404);
            }
            return Response::error('DELETE_FAILED', $e->getMessage(), [], 500);
        } catch (\Throwable $e) {
            return Response::error('DELETE_FAILED', 'Erreur inattendue.', [], 500);
        }
        return Response::ok(['deleted' => $report]);
    }
}
