<?php

declare(strict_types=1);

namespace Skyboard\Interfaces\Http\Controllers\Admin;

use Skyboard\Application\Services\Admin\UserAdminService;
use Skyboard\Infrastructure\Http\Request;
use Skyboard\Infrastructure\Http\Response;

final class UserAdminController
{
    use AdminControllerTrait;

    public function __construct(private readonly UserAdminService $users)
    {
    }

    public function list(Request $request): Response
    {
        if ($response = $this->requireAdmin($request)) {
            return $response;
        }

        return Response::ok(['users' => $this->users->list()]);
    }

    public function create(Request $request): Response
    {
        if ($response = $this->requireAdmin($request)) {
            return $response;
        }

        $body = $request->body ?? [];
        $email = (string) ($body['email'] ?? '');
        $password = (string) ($body['password'] ?? '');
        $role = (string) ($body['role'] ?? 'standard');
        $pseudo = isset($body['pseudo']) ? (string) $body['pseudo'] : null;

        try {
            $user = $this->users->create($email, $password, $role, $pseudo);
        } catch (\Throwable $e) {
            return Response::error('CREATE_FAILED', $e->getMessage(), [], 422);
        }

        return Response::ok(['user' => $user], 201);
    }

    public function update(Request $request): Response
    {
        if ($response = $this->requireAdmin($request)) {
            return $response;
        }

        $body = $request->body ?? [];
        $id = isset($body['id']) ? (int) $body['id'] : 0;
        if ($id <= 0) {
            return Response::error('INVALID_PAYLOAD', 'Données invalides.', [], 422);
        }

        try {
            $user = $this->users->updateProfile($id, $body);
        } catch (\RuntimeException $e) {
            return Response::error('NOT_FOUND', $e->getMessage(), [], 404);
        } catch (\Throwable $e) {
            return Response::error('UPDATE_FAILED', $e->getMessage(), [], 422);
        }

        return Response::ok(['user' => $user]);
    }

    public function delete(Request $request): Response
    {
        if ($response = $this->requireAdmin($request)) {
            return $response;
        }

        $id = isset($request->body['id']) ? (int) $request->body['id'] : 0;
        if ($id <= 0) {
            return Response::error('INVALID_PAYLOAD', 'Données invalides.', [], 422);
        }

        $this->users->delete($id);
        return Response::ok(['deleted' => true]);
    }

    public function data(Request $request): Response
    {
        if ($response = $this->requireAdmin($request)) {
            return $response;
        }
        $raw = $request->getAttribute('id');
        $id = is_string($raw) ? (int) $raw : (int) $raw;
        if ($id <= 0) {
            return Response::error('INVALID_PAYLOAD', 'Identifiant invalide.', [], 422);
        }
        try {
            $payload = $this->users->getData($id);
        } catch (\RuntimeException $e) {
            return Response::error('NOT_FOUND', 'Utilisateur introuvable.', [], 404);
        }
        return Response::ok($payload);
    }
}
