<?php

declare(strict_types=1);

namespace Skyboard\Interfaces\Http\Controllers;

use Skyboard\Application\Services\AuthService;
use Skyboard\Infrastructure\Http\Request;
use Skyboard\Infrastructure\Http\Response;

final class AuthController
{
    public function __construct(private readonly AuthService $auth)
    {
    }

    public function register(Request $request): Response
    {
        $email = (string) ($request->body['email'] ?? '');
        $password = (string) ($request->body['password'] ?? '');
        if (!filter_var($email, FILTER_VALIDATE_EMAIL) || strlen($password) < 8) {
            return Response::error('INVALID_PAYLOAD', 'Email ou mot de passe invalide.', [], 422);
        }
        try {
            $result = $this->auth->register($email, $password);
        } catch (\RuntimeException $e) {
            if ($e->getMessage() === 'EMAIL_ALREADY_EXISTS') {
                return Response::error('EMAIL_ALREADY_EXISTS', 'Adresse déjà utilisée.', [], 409);
            }
            throw $e;
        }
        return $this->respondWithSession($result);
    }

    public function login(Request $request): Response
    {
        $email = (string) ($request->body['email'] ?? '');
        $password = (string) ($request->body['password'] ?? '');
        if (!filter_var($email, FILTER_VALIDATE_EMAIL) || $password === '') {
            return Response::error('INVALID_PAYLOAD', 'Identifiants invalides.', [], 422);
        }
        try {
            $result = $this->auth->login($email, $password);
        } catch (\RuntimeException $e) {
            $code = strtoupper((string) $e->getMessage());
            $msg = $code === 'EMAIL_NOT_FOUND' || $code === 'INVALID_CREDENTIALS' ? 'Identifiants invalides.' : 'Connexion refusée.';
            return Response::error($code, $msg, [], 401);
        }
        return $this->respondWithSession($result);
    }

    public function logout(Request $request): Response
    {
        $session = $request->getAttribute('session');
        $cookie = 'sb_session=deleted; Path=/; Expires=Thu, 01 Jan 1970 00:00:00 GMT; SameSite=Lax; HttpOnly; Secure';
        if ($session) {
            $this->auth->logout($session['token']);
        }
        return Response::ok([], 200, ['Set-Cookie' => $cookie]);
    }

    public function session(Request $request): Response
    {
        $session = $request->getAttribute('session');
        if (!$session) {
            return Response::ok(['authenticated' => false]);
        }
        $data = $this->auth->session($session['token']);
        if (!$data) {
            return Response::ok(['authenticated' => false]);
        }
        return Response::ok([
            'authenticated' => true,
            'csrf' => $data['csrf'],
            'user' => [
                'id' => $data['user']['id'],
                'email' => $data['user']['email'],
                'role' => $data['user']['role'] ?? 'standard',
            ],
        ]);
    }

    /** @param array<string,mixed> $session */
    private function respondWithSession(array $session): Response
    {
        $payload = [
            'csrf' => $session['csrf'],
            'user' => [
                'id' => $session['user']['id'] ?? null,
                'email' => $session['user']['email'] ?? null,
                'role' => $session['user']['role'] ?? 'standard',
            ],
        ];
        $cookie = sprintf('sb_session=%s; Path=/; HttpOnly; SameSite=Lax; Secure', $session['token']);
        return Response::ok($payload, 200, ['Set-Cookie' => $cookie]);
    }
}
