<?php

declare(strict_types=1);

namespace Skyboard\Interfaces\Http\Controllers;

use Skyboard\Application\Services\NotificationRichReadService;
use Skyboard\Infrastructure\Http\Request;
use Skyboard\Infrastructure\Http\Response;

final class NotificationRichController
{
    public function __construct(private readonly NotificationRichReadService $notifications)
    {
    }

    public function list(Request $request): Response
    {
        $user = $request->getAttribute('user');
        if (!$user) {
            return Response::error('UNAUTHORIZED', 'Authentification requise.', [], 401);
        }
        $mode = (string) ($request->query['mode'] ?? 'unread');
        $payload = $this->notifications->listForUser((int) $user['id'], $mode);
        return Response::ok($payload);
    }

    public function get(Request $request): Response
    {
        $user = $request->getAttribute('user');
        if (!$user) {
            return Response::error('UNAUTHORIZED', 'Authentification requise.', [], 401);
        }
        $raw = $request->getAttribute('id');
        $id = is_string($raw) ? (int) $raw : (int) $raw;
        if ($id <= 0) {
            return Response::error('INVALID_PAYLOAD', 'Identifiant invalide.', [], 422);
        }
        $notif = $this->notifications->getOne((int) $user['id'], $id);
        if (!$notif) {
            return Response::error('NOT_FOUND', 'Notification introuvable.', [], 404);
        }
        $payload = ['item' => $notif, 'now' => time()];
        if (isset($notif['actions']) && is_array($notif['actions'])) {
            $payload['actions'] = $notif['actions'];
        }
        if (isset($notif['manifestVersion'])) $payload['manifestVersion'] = (int) $notif['manifestVersion'];
        if (isset($notif['manifestEtag'])) $payload['manifestEtag'] = (string) $notif['manifestEtag'];
        return Response::ok($payload);
    }

    public function categories(Request $request): Response
    {
        $user = $request->getAttribute('user');
        if (!$user) {
            return Response::error('UNAUTHORIZED', 'Authentification requise.', [], 401);
        }
        $list = $this->notifications->listCategoriesWithSubscriptionForUser((int) $user['id']);
        return Response::ok(['categories' => $list, 'now' => time()]);
    }
}
