<?php

declare(strict_types=1);

namespace Skyboard\Interfaces\Http\Controllers;

use Skyboard\Infrastructure\Http\Response;
use Skyboard\Infrastructure\Packs\DatasetStore;
use Skyboard\Infrastructure\Packs\RulesCatalog;
use Skyboard\Infrastructure\Packs\SystemTagRegistry;
use Skyboard\Infrastructure\Packs\UiSlotRegistry;

final class PacksController
{
    private string $enabledFile;
    private string $catalogFile;
    private string $packsDir;

    /** @var array<string, array|null> */
    private array $manifestCache = [];

    public function __construct(
        private readonly SystemTagRegistry $systemTagRegistry,
        private readonly UiSlotRegistry $uiSlotRegistry,
        private readonly DatasetStore $datasetStore,
        private readonly RulesCatalog $rulesCatalog
    ) {
        $this->enabledFile = \ROOT_PATH . '/config/packs/enabled.json';
        $this->catalogFile = \ROOT_PATH . '/config/packs/catalog.json';
        $this->packsDir = \ROOT_PATH . '/packs';
    }

    public function enabled(): Response
    {
        $state = $this->loadEnabledState();
        return Response::ok([
            'enabled' => $state['enabled'],
            'disabled' => $state['disabled'],
        ]);
    }

    public function slots(): Response
    {
        $state = $this->loadEnabledState();
        $enabledPacks = $state['enabled'];
        $payload = [];
        foreach ($this->uiSlotRegistry->getEnabledSlots($enabledPacks) as $slot) {
            $payload[] = [
                'slot' => $slot->id,
                'module' => $slot->component,
                'export' => $slot->config['export'] ?? 'default',
                'pack' => $slot->packId,
                'order' => $slot->config['order'] ?? 100,
            ];
        }
        return Response::ok(['slots' => $payload]);
    }

    public function manifest(): Response
    {
        $manifest = [
            'tags' => [],
            'rules' => [],
            'datasets' => [],
        ];

        foreach ($this->systemTagRegistry->getAll() as $tagDef) {
            [$packId] = $this->splitTagInternalKey($tagDef->internalKey);
            $manifest['tags'][] = [
                'key' => $tagDef->shortKey,
                'category' => $tagDef->category,
                'ui' => $tagDef->ui,
                'fallback' => $tagDef->fallback,
                'pack' => $packId,
            ];
        }

        foreach ($this->rulesCatalog->getAllPackRules() as $rule) {
            $manifest['rules'][] = [
                'id' => $rule->id(),
                'scope' => $rule->scope(),
                'priority' => $rule->priority(),
                'triggers' => $rule->triggers(),
                'conditions' => $rule->conditions(),
                'actions' => $rule->actions(),
                'source' => $rule->source(),
                'description' => $rule->description(),
            ];
        }

        foreach ($this->datasetStore->getAll() as $dataset) {
            $manifest['datasets'][] = [
                'id' => $dataset->id,
                'packId' => $dataset->packId,
                'kind' => $dataset->kind,
                'schema' => $dataset->schema,
            ];
        }

        $manifest['ok'] = true;
        return Response::ok($manifest);
    }

    public function config(): Response
    {
        $state = $this->loadEnabledState();
        $catalog = $this->loadCatalog();
        $packs = [];
        $seen = [];

        foreach ($catalog['entries'] as $entry) {
            $packs[] = $this->buildPackPayload($entry, $state);
            $seen[] = $entry['id'];
        }

        foreach ($this->collectUnknownPackIds($state, $seen) as $packId) {
            $fallbackEntry = [
                'id' => $packId,
                'name' => ucfirst($packId),
                'descriptionHtml' => '<p>Pack détecté sans fiche catalogue.</p>',
                'highlights' => [],
                'availability' => [
                    'status' => 'unknown',
                    'requiresLicense' => false,
                    'defaultEnabled' => false,
                ],
            ];
            $packs[] = $this->buildPackPayload($fallbackEntry, $state);
        }

        return Response::ok([
            'packs' => $packs,
            'meta' => [
                'catalogRevision' => $catalog['revision'],
                'generatedAt' => gmdate('c'),
            ],
        ]);
    }

    public function updateConfig(array $data): Response
    {
        $toggleMap = $this->extractToggleMap($data);
        if ($toggleMap === null || $toggleMap === []) {
            return Response::error('INVALID_PAYLOAD', 'Aucun toggle reçu.', [], 400);
        }

        $state = $this->loadEnabledState();
        $catalog = $this->loadCatalog();
        $catalogIndex = [];
        foreach ($catalog['entries'] as $entry) {
            $catalogIndex[$entry['id']] = $entry;
        }

        $error = $this->applyToggles($state, $toggleMap, $catalogIndex);
        if ($error !== null) {
            $status = isset($error['status']) ? (int) $error['status'] : 422;
            $code = is_string($error['code'] ?? null) ? (string) $error['code'] : 'INVALID_PAYLOAD';
            $message = is_string($error['message'] ?? null) ? (string) $error['message'] : 'Requête invalide.';
            $details = [];
            foreach (['pack','access'] as $k) {
                if (array_key_exists($k, $error)) {
                    $details[$k] = $error[$k];
                }
            }
            return Response::error($code, $message, $details, $status);
        }

        if (!$this->persistEnabledState($state)) {
            return Response::error('CONFIG_WRITE_FAILED', 'Impossible de sauvegarder la configuration des packs.', [], 500);
        }
        return $this->config();
    }

    public function getDataset(string $id): Response
    {
        $data = $this->datasetStore->load($id);
        $etag = $this->datasetStore->getEtag($id);
        if ($data === null) {
            return Response::error('DATASET_NOT_FOUND', 'Dataset introuvable.', [], 404);
        }
        $headers = [];
        if ($etag !== '') {
            $headers['ETag'] = $etag;
        }
        return Response::ok([
            'data' => $data,
            'etag' => $etag,
        ], 200, $headers);
    }

    // Removed: updateDataset migrated to /api/commands (Non-Board Bus)

    // Removed: deleteDatasetKey migrated to /api/commands (Non-Board Bus)

    /**
     * @return array{enabled:list<string>,disabled:list<string>}
     */
    private function loadEnabledState(): array
    {
        if (!is_file($this->enabledFile)) {
            return ['enabled' => [], 'disabled' => []];
        }
        $raw = file_get_contents($this->enabledFile);
        if ($raw === false) {
            return ['enabled' => [], 'disabled' => []];
        }
        $json = json_decode($raw, true);
        if (!is_array($json)) {
            return ['enabled' => [], 'disabled' => []];
        }

        $enabled = [];
        if (isset($json['enabled']) && is_array($json['enabled'])) {
            foreach ($json['enabled'] as $packId) {
                if (is_string($packId) && $packId !== '') {
                    $enabled[] = $packId;
                }
            }
        }

        $disabled = [];
        if (isset($json['disabled']) && is_array($json['disabled'])) {
            foreach ($json['disabled'] as $packId) {
                if (is_string($packId) && $packId !== '') {
                    $disabled[] = $packId;
                }
            }
        }

        $enabled = array_values(array_unique($enabled));
        sort($enabled);
        $disabled = array_values(array_unique(array_diff($disabled, $enabled)));
        sort($disabled);

        return ['enabled' => $enabled, 'disabled' => $disabled];
    }

    private function persistEnabledState(array $state): bool
    {
        $payload = [
            'enabled' => array_values($state['enabled']),
            'disabled' => array_values($state['disabled']),
        ];
        $json = json_encode($payload, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES);
        if ($json === false) {
            return false;
        }
        $json .= PHP_EOL;
        $dir = dirname($this->enabledFile);
        if (!is_dir($dir)) {
            @mkdir($dir, 0775, true);
        }
        return @file_put_contents($this->enabledFile, $json, LOCK_EX) !== false;
    }

    /**
     * @return array{entries:list<array<string,mixed>>,revision:string}
     */
    private function loadCatalog(): array
    {
        if (!is_file($this->catalogFile)) {
            return ['entries' => [], 'revision' => 'missing'];
        }
        $raw = file_get_contents($this->catalogFile);
        if ($raw === false) {
            return ['entries' => [], 'revision' => 'missing'];
        }

        $json = json_decode($raw, true);
        $entries = [];
        if (is_array($json) && isset($json['packs']) && is_array($json['packs'])) {
            foreach ($json['packs'] as $pack) {
                if (is_array($pack) && isset($pack['id']) && is_string($pack['id']) && $pack['id'] !== '') {
                    $entries[] = $pack;
                }
            }
        }

        $revision = substr(sha1($raw), 0, 12);
        return ['entries' => $entries, 'revision' => $revision];
    }

    /**
     * @param array{enabled:list<string>,disabled:list<string>} $state
     * @param list<string> $known
     * @return list<string>
     */
    private function collectUnknownPackIds(array $state, array $known): array
    {
        $all = array_unique(array_merge($state['enabled'], $state['disabled']));
        $extra = array_values(array_diff($all, $known));
        sort($extra);
        return $extra;
    }

    /**
     * @param array<string,mixed> $entry
     * @param array{enabled:list<string>,disabled:list<string>} $state
     */
    private function buildPackPayload(array $entry, array $state): array
    {
        $packId = isset($entry['id']) && is_string($entry['id']) ? $entry['id'] : '';
        if ($packId === '') {
            return [
                'id' => '',
                'name' => 'unknown',
                'enabled' => false,
                'installed' => false,
                'status' => 'unknown',
                'access' => [
                    'locked' => true,
                    'reason' => 'invalid',
                    'status' => 'unknown',
                    'requiresLicense' => false,
                    'defaultEnabled' => false,
                    'licenseTier' => null,
                    'licenseNotes' => null,
                ],
                'technical' => $this->formatTechnical(null),
                'metadata' => ['availability' => [], 'licensing' => []],
                'tagline' => null,
                'descriptionHtml' => '',
                'highlights' => [],
            ];
        }

        $manifest = $this->loadManifestForPack($packId);
        $isEnabled = in_array($packId, $state['enabled'], true);
        $access = $this->computeAccessState($entry, $manifest, $isEnabled);

        $status = $isEnabled ? 'active' : ($manifest !== null ? 'available' : 'unavailable');
        $availabilityStatus = $entry['availability']['status'] ?? null;
        if (!$isEnabled && is_string($availabilityStatus) && $availabilityStatus !== '') {
            $status = $availabilityStatus;
        }

        return [
            'id' => $packId,
            'name' => isset($entry['name']) && is_string($entry['name']) ? $entry['name'] : $packId,
            'tagline' => isset($entry['tagline']) && is_string($entry['tagline']) ? $entry['tagline'] : null,
            'descriptionHtml' => isset($entry['descriptionHtml']) && is_string($entry['descriptionHtml']) ? $entry['descriptionHtml'] : '',
            'highlights' => $this->normalizeHighlights($entry['highlights'] ?? []),
            'enabled' => $isEnabled,
            'installed' => $manifest !== null,
            'status' => $status,
            'access' => $access,
            'technical' => $this->formatTechnical($manifest),
            'metadata' => [
                'availability' => $entry['availability'] ?? [],
                'licensing' => $entry['licensing'] ?? [],
            ],
        ];
    }

    /**
     * @param mixed $highlights
     * @return list<array<string,?string>>
     */
    private function normalizeHighlights(mixed $highlights): array
    {
        if (!is_array($highlights)) {
            return [];
        }
        $normalized = [];
        foreach ($highlights as $highlight) {
            if (!is_array($highlight)) {
                continue;
            }
            $normalized[] = [
                'icon' => isset($highlight['icon']) && is_string($highlight['icon']) ? $highlight['icon'] : null,
                'title' => isset($highlight['title']) && is_string($highlight['title']) ? $highlight['title'] : null,
                'description' => isset($highlight['description']) && is_string($highlight['description']) ? $highlight['description'] : null,
            ];
        }
        return $normalized;
    }

    /**
     * @param array<string,mixed> $entry
     */
    private function computeAccessState(array $entry, ?array $manifest, bool $isEnabled): array
    {
        $availability = isset($entry['availability']) && is_array($entry['availability']) ? $entry['availability'] : [];
        $status = isset($availability['status']) && is_string($availability['status']) ? $availability['status'] : 'unknown';
        $requiresLicense = (bool) ($availability['requiresLicense'] ?? false);
        $defaultEnabled = (bool) ($availability['defaultEnabled'] ?? false);
        $license = isset($entry['licensing']) && is_array($entry['licensing']) ? $entry['licensing'] : [];
        $licenseTier = isset($license['tier']) && is_string($license['tier']) ? $license['tier'] : null;
        $licenseNotes = isset($license['notes']) && is_string($license['notes']) ? $license['notes'] : null;

        $locked = false;
        $reason = null;

        if (!$isEnabled) {
            if ($manifest === null) {
                $locked = true;
                $reason = 'not-installed';
            } elseif (!in_array($status, ['available', 'beta'], true)) {
                $locked = true;
                $reason = $status === '' ? 'unavailable' : $status;
            }
        }

        return [
            'locked' => $locked,
            'reason' => $reason,
            'status' => $status,
            'requiresLicense' => $requiresLicense,
            'defaultEnabled' => $defaultEnabled,
            'licenseTier' => $licenseTier,
            'licenseNotes' => $licenseNotes,
        ];
    }

    private function formatTechnical(?array $manifest): array
    {
        if ($manifest === null) {
            return [
                'pack' => null,
                'tags' => [],
                'rules' => [],
                'datasets' => [],
                'slots' => [],
            ];
        }

        $packMeta = [];
        if (isset($manifest['pack']) && is_array($manifest['pack'])) {
            $packMeta = [
                'id' => $manifest['pack']['id'] ?? null,
                'version' => $manifest['pack']['version'] ?? null,
                'engine' => $manifest['pack']['engine'] ?? null,
            ];
        }

        $tags = [];
        foreach ($manifest['tags'] ?? [] as $tag) {
            if (!is_array($tag) || !isset($tag['key'])) {
                continue;
            }
            $ui = isset($tag['ui']) && is_array($tag['ui']) ? $tag['ui'] : [];
            $badge = isset($ui['badge']) && is_array($ui['badge']) ? $ui['badge'] : [];
            $tagList = isset($ui['tagList']) && is_array($ui['tagList']) ? $ui['tagList'] : [];
            $tags[] = [
                'key' => (string) $tag['key'],
                'category' => isset($tag['category']) && is_string($tag['category']) ? $tag['category'] : null,
                'label' => isset($badge['label']) && is_string($badge['label']) ? $badge['label'] : null,
                'icon' => isset($badge['icon']) && is_string($badge['icon']) ? $badge['icon'] : null,
                'color' => isset($badge['color']) && is_string($badge['color']) ? $badge['color'] : null,
                'hiddenInPicker' => (bool) ($tagList['hidden'] ?? false),
            ];
        }

        $rules = [];
        foreach ($manifest['rules'] ?? [] as $rule) {
            if (!is_array($rule) || !isset($rule['id'])) {
                continue;
            }
            $trigger = $rule['trigger']['on'] ?? null;
            if (is_array($rule['trigger'] ?? null) && isset($rule['trigger']['on']) && is_string($rule['trigger']['on'])) {
                $trigger = $rule['trigger']['on'];
            } elseif (isset($rule['trigger']) && is_string($rule['trigger'])) {
                $trigger = $rule['trigger'];
            }
            $rules[] = [
                'id' => (string) $rule['id'],
                'scope' => isset($rule['strata']) && is_string($rule['strata']) ? $rule['strata'] : null,
                'priority' => isset($rule['priority']) ? (int) $rule['priority'] : null,
                'trigger' => is_string($trigger) ? $trigger : null,
                'conditionsCount' => isset($rule['when']) && is_array($rule['when']) ? count($rule['when']) : 0,
                'actionsCount' => isset($rule['then']) && is_array($rule['then']) ? count($rule['then']) : 0,
            ];
        }

        $datasets = [];
        foreach ($manifest['datasets'] ?? [] as $dataset) {
            if (!is_array($dataset) || !isset($dataset['id'])) {
                continue;
            }
            $datasets[] = [
                'id' => (string) $dataset['id'],
                'kind' => isset($dataset['kind']) && is_string($dataset['kind']) ? $dataset['kind'] : 'json',
                'path' => isset($dataset['path']) && is_string($dataset['path']) ? $dataset['path'] : null,
                'schema' => $dataset['schema'] ?? null,
            ];
        }

        $slots = [];
        $ui = isset($manifest['ui']) && is_array($manifest['ui']) ? $manifest['ui'] : [];
        foreach ($ui['slots'] ?? [] as $slot) {
            if (!is_array($slot) || !isset($slot['id']) || !isset($slot['component'])) {
                continue;
            }
            $slots[] = [
                'id' => (string) $slot['id'],
                'component' => (string) $slot['component'],
                'type' => isset($slot['type']) && is_string($slot['type']) ? $slot['type'] : null,
            ];
        }

        return [
            'pack' => $packMeta,
            'tags' => $tags,
            'rules' => $rules,
            'datasets' => $datasets,
            'slots' => $slots,
        ];
    }

    /**
     * @param array<string,bool> $toggleMap
     * @param array<string,array<string,mixed>> $catalogIndex
     */
    private function applyToggles(array &$state, array $toggleMap, array $catalogIndex): ?array
    {
        $enabled = $state['enabled'];
        $disabled = $state['disabled'];

        foreach ($toggleMap as $packId => $desiredState) {
            $packId = (string) $packId;
            $entry = $catalogIndex[$packId] ?? ['id' => $packId, 'name' => $packId];
            $manifest = $this->loadManifestForPack($packId);
            $access = $this->computeAccessState($entry, $manifest, in_array($packId, $enabled, true));

            if ($desiredState && $access['locked'] === true) {
                return [
                    'ok' => false,
                    'status' => 422,
                    'code' => 'PACK_NOT_AVAILABLE',
                    'message' => sprintf("Le pack %s n'est pas disponible à l'activation.", $entry['name'] ?? $packId),
                    'pack' => $packId,
                    'access' => $access,
                ];
            }

            if ($desiredState) {
                if (!in_array($packId, $enabled, true)) {
                    $enabled[] = $packId;
                }
                $disabled = array_values(array_diff($disabled, [$packId]));
            } else {
                $enabled = array_values(array_diff($enabled, [$packId]));
                if (!in_array($packId, $disabled, true)) {
                    $disabled[] = $packId;
                }
            }
        }

        $enabled = array_values(array_unique($enabled));
        sort($enabled);
        $disabled = array_values(array_unique(array_diff($disabled, $enabled)));
        sort($disabled);

        $state['enabled'] = $enabled;
        $state['disabled'] = $disabled;

        return null;
    }

    private function extractToggleMap(array $data): ?array
    {
        $raw = $data['packs'] ?? $data['toggles'] ?? null;
        if ($raw === null || !is_array($raw)) {
            return null;
        }

        $map = [];
        if ($this->isAssoc($raw)) {
            foreach ($raw as $packId => $value) {
                if (!is_string($packId) || $packId === '') {
                    continue;
                }
                $bool = $this->normalizeBool($value);
                if ($bool === null) {
                    continue;
                }
                $map[$packId] = $bool;
            }
            return $map;
        }

        foreach ($raw as $entry) {
            if (!is_array($entry)) {
                continue;
            }
            $packId = $entry['id'] ?? $entry['pack'] ?? $entry['packId'] ?? null;
            if (!is_string($packId) || $packId === '') {
                continue;
            }
            $bool = $this->normalizeBool($entry['enabled'] ?? $entry['on'] ?? $entry['active'] ?? null);
            if ($bool === null) {
                continue;
            }
            $map[$packId] = $bool;
        }

        return $map;
    }

    private function normalizeBool(mixed $value): ?bool
    {
        if (is_bool($value)) {
            return $value;
        }
        if (is_int($value)) {
            return $value !== 0;
        }
        if (is_float($value)) {
            return $value != 0.0;
        }
        if (is_string($value)) {
            $normalized = strtolower(trim($value));
            return match ($normalized) {
                '1', 'true', 'on', 'yes' => true,
                '0', 'false', 'off', 'no' => false,
                default => null,
            };
        }
        return null;
    }

    private function isAssoc(array $array): bool
    {
        return array_keys($array) !== range(0, count($array) - 1);
    }

    private function loadManifestForPack(string $packId): ?array
    {
        if (array_key_exists($packId, $this->manifestCache)) {
            return $this->manifestCache[$packId];
        }
        $path = $this->packsDir . '/' . $packId . '/manifest.mcc.json';
        if (!is_file($path)) {
            return $this->manifestCache[$packId] = null;
        }
        $raw = file_get_contents($path);
        if ($raw === false) {
            return $this->manifestCache[$packId] = null;
        }
        $json = json_decode($raw, true);
        if (!is_array($json)) {
            return $this->manifestCache[$packId] = null;
        }
        return $this->manifestCache[$packId] = $json;
    }

    /**
     * @return array{0:string,1:string}
     */
    private function splitTagInternalKey(string $internalKey): array
    {
        $parts = explode(':', $internalKey, 2);
        if (count($parts) === 2) {
            return [$parts[0], $parts[1]];
        }
        return ['', $internalKey];
    }
}
