<?php

declare(strict_types=1);

namespace Skyboard\Interfaces\Http\Controllers;

use Skyboard\Application\Services\UserFileException;
use Skyboard\Application\Services\UserFileService;
use Skyboard\Domain\UserFiles\UserFile;
use Skyboard\Infrastructure\Http\Request;
use Skyboard\Infrastructure\Http\Response;

final class UserFilesController
{
    public function __construct(private readonly UserFileService $files)
    {
    }

    public function index(Request $request): Response
    {
        $user = $request->getAttribute('user');
        if (!$user) {
            return Response::error('UNAUTHORIZED', 'Authentification requise.', [], 401);
        }
        $list = $this->files->listForUser((int) $user['id']);
        return Response::ok(['files' => $list]);
    }

    public function folders(Request $request): Response
    {
        $user = $request->getAttribute('user');
        if (!$user) {
            return Response::error('UNAUTHORIZED', 'Authentification requise.', [], 401);
        }
        try {
            $folders = $this->files->listFolders((int) $user['id']);
            return Response::ok(['folders' => $folders]);
        } catch (\Throwable $e) {
            return Response::error('FOLDER_LIST_FAILED', 'Impossible de lister les dossiers.', [], 500);
        }
    }

    // Removed: upload/compose/updateContent/rename/delete endpoints migrated to /api/commands (Non-Board Bus)

    public function content(Request $request): Response
    {
        $user = $request->getAttribute('user');
        if (!$user) {
            return Response::error('UNAUTHORIZED', 'Authentification requise.', [], 401);
        }
        $publicId = (string) $request->getAttribute('id', '');
        if ($publicId === '') {
            return Response::error('INVALID_FILE_ID', 'Identifiant de fichier invalide.', [], 422);
        }

        $headers = array_change_key_case($request->headers, CASE_LOWER);
        $accept = (string) ($headers['accept'] ?? '');
        $format = strtolower((string) ($request->query['format'] ?? ''));
        $wantsJson = $format === 'json'
            || str_contains($accept, 'application/json')
            || str_contains($accept, 'application/*+json');

        if ($wantsJson) {
            try {
                $result = $this->files->readTextFile((int) $user['id'], $publicId);
                return Response::ok(['file' => $result['file'], 'content' => $result['content']]);
            } catch (UserFileException $exception) {
                return $this->error($exception);
            }
        }

        $resolved = $this->files->resolveForDownload((int) $user['id'], $publicId);
        if ($resolved === null) {
            return Response::error('FILE_NOT_FOUND', 'Fichier introuvable.', [], 404);
        }

        /** @var UserFile $file */
        $file = $resolved['file'];
        $path = $resolved['path'];
        $body = @file_get_contents($path);
        if ($body === false) {
            return Response::error('FILE_NOT_FOUND', 'Fichier introuvable.', [], 404);
        }

        $headers = [
            'Content-Disposition' => $this->buildContentDisposition($file),
            'Cache-Control' => 'private, max-age=31536000, immutable',
        ];
        if ($file->checksum()) {
            $headers['ETag'] = '"' . $file->checksum() . '"';
        }
        return Response::binary($body, $file->mimeType() ?? 'application/octet-stream', 200, $headers);
    }

    private function error(UserFileException $exception): Response
    {
        $details = $exception->details();
        return Response::error($exception->errorCode(), $exception->getMessage(), $details, $exception->status());
    }

    private function buildContentDisposition(UserFile $file): string
    {
        $fallback = $file->publicId();
        $name = $this->files->buildDownloadName($file);
        $ascii = $this->stripInvalidHeaderChars($name);
        if ($ascii === '') {
            $ascii = $fallback;
        }
        $encoded = rawurlencode($name);
        $escapedAscii = addcslashes($ascii, '"\\');
        return sprintf('inline; filename="%s"; filename*=UTF-8\'\'%s', $escapedAscii, $encoded);
    }

    private function stripInvalidHeaderChars(string $value): string
    {
        $sanitized = preg_replace('/[\\x00-\\x1f\\x7f]+/u', '', $value) ?? '';
        $sanitized = str_replace(["\r", "\n"], '', $sanitized);
        return trim($sanitized);
    }
}
