<?php

declare(strict_types=1);

namespace Skyboard\Interfaces\Http\Middleware;

use Closure;
use Skyboard\Infrastructure\Http\Request;
use Skyboard\Infrastructure\Http\Response;
use Skyboard\Infrastructure\Security\RateLimiterInterface;

final class RateLimitMiddleware
{
    public function __construct(
        private readonly RateLimiterInterface $limiter,
        private readonly int $threshold = 10,
        private readonly int $banSeconds = 86400
    ) {
    }

    public function __invoke(Request $request, Closure $next): Response
    {
        $ip = $request->server['REMOTE_ADDR'] ?? '0.0.0.0';
        if ($this->limiter->isBanned($ip)) {
            return Response::error('RATE_LIMITED', 'Trop de requêtes depuis cette adresse IP.', [], 429);
        }
        $response = $next($request);
        if (in_array($request->path, ['/api/auth/login', '/api/auth/register'], true) && $response->status >= 400) {
            $this->limiter->hit($ip, $this->threshold, $this->banSeconds);
        } elseif ($request->path === '/api/auth/login' && $response->status === 200) {
            $this->limiter->clear($ip);
        }
        return $response;
    }
}
