#!/usr/bin/env bash
set -euo pipefail

# Ensure we run from repo root (one level up from this script dir)
SCRIPT_DIR="$(cd "$(dirname "$0")" && pwd)"
cd "${SCRIPT_DIR}/../.."

echo "[guards] enforcing canonical errors in controllers"

violations=0

# Fail if non-canonical error shapes remain in controllers
if rg -uu -n "Response::json\(\s*\[\s*'error'\s*=>" src/Interfaces/Http/Controllers >/dev/null; then
  echo "[guards][FAIL] Found legacy Response::json([... 'error' => ...]) in controllers"
  rg -uu -n "Response::json\(\s*\[\s*'error'\s*=>" src/Interfaces/Http/Controllers || true
  violations=$((violations+1))
fi

# Fail if legacy success payloads are used
if rg -uu -n "Response::json\(\[\s*'status'\s*=>" src/Interfaces/Http/Controllers >/dev/null; then
  echo "[guards][FAIL] Found legacy Response::json([... 'status' => ...]) patterns; use Response::ok(...)"
  rg -uu -n "Response::json\(\[\s*'status'\s*=>" src/Interfaces/Http/Controllers || true
  violations=$((violations+1))
fi

if [ "$violations" -gt 0 ]; then
  echo "[guards] violations=$violations"
  exit 1
fi

# Routes guard: forbid mutating routes outside the canonical authorization list
echo "[guards] scanning mutating routes (public/index.php)"
mutants=$(awk '/\$router->add\(/ {print NR ":" $0}' public/index.php | \
  rg "\$router->add\('(POST|PUT|PATCH|DELETE)'\s*,\s*'([^"]+)'" -n -S || true)

forbidden=0
while IFS= read -r line; do
  [ -z "$line" ] && continue
  verb=$(echo "$line" | sed -E "s/.*add\('\(POST|PUT|PATCH|DELETE\)'.*/\1/")
  path=$(echo "$line" | sed -E "s/.*'\/(api[^']*)'.*/\/\1/")
  allow=0
  if [ "$verb" = "POST" ] && [ "$path" = "/api/commands" ]; then allow=1; fi
  if [ "$verb" = "POST" ] && echo "$path" | rg -q '^/api/admin/'; then allow=1; fi
  if [ "$verb" = "POST" ] && { [ "$path" = "/api/auth/login" ] || [ "$path" = "/api/auth/register" ]; }; then allow=1; fi
  if [ "$verb" = "POST" ] && [ "$path" = "/api/dev/elevate-admin" ]; then allow=1; fi
  if [ "$verb" = "PUT" ] && [ "$path" = "/api/packs/config" ]; then allow=1; fi
  if [ "$allow" -eq 0 ]; then
    echo "[guards][FAIL] Forbidden route: $line"
    forbidden=$((forbidden+1))
  fi
done <<< "$mutants"

if [ "$forbidden" -gt 0 ]; then
  echo "[guards] forbidden routes: $forbidden"
  exit 1
fi

echo "[guards] OK"
exit 0
