#!/usr/bin/env php
<?php

declare(strict_types=1);

use Skyboard\Infrastructure\Persistence\DatabaseConnection;
use Skyboard\Application\Services\Scheduler\Slot;

require __DIR__ . '/../../bootstrap.php';

// Basic CLI runner for notifications digest (BROADCAST & PERSONALIZED)
// - Timezone logic: Europe/Paris → UTC storage
// - Idempotent upserts; global + per-category advisory locks

const TZ = 'Europe/Paris';

// CLI options (very light parsing)
$argv = $argv ?? [];
$dryRun = in_array('--dry-run', $argv, true);

function log_line(string $msg): void {
    $ts = (new DateTimeImmutable('now', new DateTimeZone('UTC')))->format('c');
    fwrite(STDOUT, "[$ts] $msg\n");
}

// next_slot now delegated to Slot::nextSlot

function get_lock(PDO $pdo, string $key): bool {
    $stmt = $pdo->prepare('SELECT GET_LOCK(:k, 0) AS ok');
    $stmt->execute(['k' => $key]);
    $row = $stmt->fetch(PDO::FETCH_ASSOC) ?: [];
    return (int) ($row['ok'] ?? 0) === 1;
}

function release_lock(PDO $pdo, string $key): void {
    try {
        $stmt = $pdo->prepare('SELECT RELEASE_LOCK(:k)');
        $stmt->execute(['k' => $key]);
    } catch (Throwable) {}
}

// Boot
$config = \Skyboard\config();
$conn = new DatabaseConnection($config['dsn'], $config['db_user'] ?? null, $config['db_password'] ?? null);
$pdo = $conn->pdo();

$now = time();
log_line('scheduler start now=' . $now . ($dryRun ? ' [DRY-RUN]' : ''));

// Global lock to prevent concurrent runs
if (!get_lock($pdo, 'sched_notifications_digest_global')) {
    log_line('skip global lock — another run is in progress');
    exit(0);
}

// BROADCAST digest
$qCat = $pdo->query("SELECT id, audience_mode, dispatch_mode, frequency_kind, frequency_param, anchor_ts FROM notification_categories WHERE active=1 AND dispatch_mode='BROADCAST' AND frequency_kind <> 'IMMEDIATE'");
$cats = $qCat ? $qCat->fetchAll(PDO::FETCH_ASSOC) : [];
foreach ($cats as $cat) {
    $catId = (int) $cat['id'];
    $lockKey = 'sched_broadcast_cat_' . $catId;
    if (!get_lock($pdo, $lockKey)) {
        log_line("skip lock cat=$catId");
        continue;
    }
    try {
        $kind = (string) $cat['frequency_kind'];
        $param = isset($cat['frequency_param']) ? (int) $cat['frequency_param'] : null;
        $anchor = isset($cat['anchor_ts']) ? (int) $cat['anchor_ts'] : null;
        $slot = Slot::nextSlot($now, $kind, $param, $anchor, TZ);
        if ($slot === null || $now < $slot) {
            log_line("broadcast cat=$catId slot_wait slot=$slot");
            continue;
        }
        // Last slot cursor
        $stmt = $pdo->prepare('SELECT last_slot_ts FROM scheduler_meta WHERE category_id = :cat');
        $stmt->execute(['cat' => $catId]);
        $lastRow = $stmt->fetch(PDO::FETCH_ASSOC);
        $last = $lastRow ? (int) $lastRow['last_slot_ts'] : 0;
        if ($last >= $slot) {
            log_line("broadcast cat=$catId slot_done last=$last slot=$slot");
            continue;
        }
        $qNotif = $pdo->prepare('SELECT id FROM notifications_rich WHERE category_id = :cat AND active=1 AND created_at > :last ORDER BY created_at ASC, id ASC');
        $qNotif->execute(['cat' => $catId, 'last' => $last]);
        $notifs = $qNotif->fetchAll(PDO::FETCH_ASSOC) ?: [];
        $delta = count($notifs);
        $aud = strtoupper((string) ($cat['audience_mode'] ?? 'EVERYONE'));
        $insCount = 0;
        foreach ($notifs as $n) {
            $nid = (int) $n['id'];
            if ($aud === 'EVERYONE') {
                $sql = 'INSERT INTO notification_user_state(user_id, notification_id, available_at)
                        SELECT u.id, :nid, :slot FROM users u
                        ON DUPLICATE KEY UPDATE available_at = LEAST(VALUES(available_at), notification_user_state.available_at)';
                $ins = $pdo->prepare($sql);
                if ($dryRun) { log_line("[DRY] upsert NUS everyone nid=$nid slot=$slot"); } else { $ins->execute(['nid' => $nid, 'slot' => $slot]); }
            } else {
                $sql = 'INSERT INTO notification_user_state(user_id, notification_id, available_at)
                        SELECT us.user_id, :nid, :slot FROM user_subscriptions us
                        WHERE us.category_id = :cat AND us.subscribed = 1
                        ON DUPLICATE KEY UPDATE available_at = LEAST(VALUES(available_at), notification_user_state.available_at)';
                $ins = $pdo->prepare($sql);
                if ($dryRun) { log_line("[DRY] upsert NUS subscribers nid=$nid cat=$catId slot=$slot"); } else { $ins->execute(['nid' => $nid, 'slot' => $slot, 'cat' => $catId]); }
            }
            $insCount++;
        }
        $up = $pdo->prepare('INSERT INTO scheduler_meta(category_id, last_slot_ts) VALUES(:cat, :slot)
                              ON DUPLICATE KEY UPDATE last_slot_ts = VALUES(last_slot_ts)');
        if ($dryRun) { log_line("[DRY] update scheduler_meta cat=$catId last_slot_ts=$slot"); } else { $up->execute(['cat' => $catId, 'slot' => $slot]); }
        log_line("broadcast cat=$catId slot=$slot delta=$delta upserts=$insCount status=OK");
    } catch (Throwable $e) {
        log_line("broadcast cat=$catId status=ERR msg=" . $e->getMessage());
    } finally {
        release_lock($pdo, $lockKey);
    }
}

// PERSONALIZED progression
$qSub = $pdo->query("SELECT us.user_id, us.category_id, us.override_kind, us.override_param, us.cycle_anchor_ts, us.last_delivered_notification_id,
                            c.frequency_kind, c.frequency_param, c.anchor_ts, c.allow_user_override
                     FROM user_subscriptions us
                     JOIN notification_categories c ON c.id = us.category_id
                     WHERE us.subscribed = 1 AND c.active = 1 AND c.dispatch_mode = 'PERSONALIZED'");
$subs = $qSub ? $qSub->fetchAll(PDO::FETCH_ASSOC) : [];
foreach ($subs as $row) {
    $userId = (int) $row['user_id'];
    $catId = (int) $row['category_id'];
    $lockKey = 'sched_personal_cat_' . $catId;
    if (!get_lock($pdo, $lockKey)) {
        log_line("skip lock personalized cat=$catId");
        continue;
    }
    try {
        $allowOverride = !empty($row['allow_user_override']);
        $okOverride = $allowOverride && !empty($row['override_kind']);
        $kind = strtoupper($okOverride ? (string) $row['override_kind'] : (string) $row['frequency_kind']);
        $param = $okOverride ? (isset($row['override_param']) ? (int) $row['override_param'] : null) : (isset($row['frequency_param']) ? (int) $row['frequency_param'] : null);
        $anchor = isset($row['cycle_anchor_ts']) && (int)$row['cycle_anchor_ts'] > 0
            ? (int) $row['cycle_anchor_ts']
            : (isset($row['anchor_ts']) ? (int) $row['anchor_ts'] : null);
        // IMMEDIATE interdit avec PERSONALIZED (validé côté Admin); ignorer si rencontré
        if ($kind === 'IMMEDIATE') {
            continue;
        }
        $slot = Slot::nextSlot($now, $kind, $param, $anchor, TZ);
        if ($slot === null || $now < $slot) {
            continue;
        }
        $lastDelivered = isset($row['last_delivered_notification_id']) ? (int) $row['last_delivered_notification_id'] : 0;
        if ($lastDelivered > 0) {
            // find next after last
            $stmt = $pdo->prepare('SELECT nr2.id FROM notifications_rich nr1 JOIN notifications_rich nr2 ON nr2.category_id = nr1.category_id
                                   WHERE nr1.id = :last AND nr2.active = 1 AND nr2.category_id = :cat AND nr2.sequence_index > nr1.sequence_index
                                   ORDER BY nr2.sequence_index ASC LIMIT 1');
            $stmt->execute(['last' => $lastDelivered, 'cat' => $catId]);
            $next = $stmt->fetch(PDO::FETCH_ASSOC);
            if (!$next) {
                // No further episode; skip
                continue;
            }
            $nextId = (int) $next['id'];
        } else {
            // first episode
            $stmt = $pdo->prepare('SELECT id FROM notifications_rich WHERE category_id = :cat AND active = 1 ORDER BY sequence_index ASC LIMIT 1');
            $stmt->execute(['cat' => $catId]);
            $next = $stmt->fetch(PDO::FETCH_ASSOC);
            if (!$next) continue;
            $nextId = (int) $next['id'];
        }

        // Upsert the NUS
        $ins = $pdo->prepare('INSERT INTO notification_user_state(user_id, notification_id, available_at) VALUES(:user, :nid, :slot)
                              ON DUPLICATE KEY UPDATE available_at = LEAST(VALUES(available_at), notification_user_state.available_at)');
        if ($dryRun) { log_line("[DRY] personalized NUS user=$userId nid=$nextId slot=$slot"); } else { $ins->execute(['user' => $userId, 'nid' => $nextId, 'slot' => $slot]); }

        // Update last delivered
        $upd = $pdo->prepare('UPDATE user_subscriptions SET last_delivered_notification_id = :nid WHERE user_id = :user AND category_id = :cat');
        if ($dryRun) { log_line("[DRY] update last_delivered user=$userId cat=$catId nid=$nextId"); } else { $upd->execute(['nid' => $nextId, 'user' => $userId, 'cat' => $catId]); }
        log_line("personalized user=$userId cat=$catId slot=$slot delivered=$nextId status=OK");
    } catch (Throwable $e) {
        log_line("personalized user=$userId cat=$catId status=ERR msg=" . $e->getMessage());
    } finally {
        release_lock($pdo, $lockKey);
    }
}

log_line('scheduler end');
release_lock($pdo, 'sched_notifications_digest_global');
